
require 'sketchup.rb'
require 'Phlatboyz/Constants.rb'
require 'Phlatboyz/PhlatEdge.rb'
require 'Phlatboyz/tools/OffsetCut.rb'

module PhlatScript

  class CutTool < PhlatTool
    @offset_dist = 0.1.inch
    @switch_edge_side = false

    def initialize
      super()
      @tooltype = (PB_MENU_MENU | PB_MENU_TOOLBAR | PB_MENU_CONTEXT)
      @@N = 0
      @active_face = nil
      @bit_diameter = Sketchup.active_model.get_attribute $dict_name, $dict_bit_diameter, $default_bit_diameter
    end

    def activate
      super
      @bit_diameter = Sketchup.active_model.get_attribute $dict_name, $dict_bit_diameter, $default_bit_diameter
    end

    def onContextMenu(menuItem)
      cut_class.cut(Sketchup.active_model.selection) 
    end

    def onMouseMove(flags, x, y, view)
      @ip.pick view, x, y
      @active_face = activeFaceFromInputPoint(@ip)

      if(@active_face)
        view.tooltip = @ip.tooltip
        #Sketchup::set_status_text @ip.tooltip, SB_VCB_LABEL
      end
      view.invalidate #if(@ip.display?)
    end

    def onLButtonDown(flags, x, y, view)
      @ip.pick view, x, y
      @active_face = activeFaceFromInputPoint(@ip)

      if(@active_face)
        cut_face(@active_face, self.offset)
      end
      self.reset(view)

      # Clear any inference lock
      view.lock_inference
    end

    def onKeyDown(key, repeat, flags, view)
      if key == CONSTRAIN_MODIFIER_KEY
        @switch_edge_side = true
        view.invalidate
      elsif key == 78
        @@N += 1
        @active_face = activeFaceFromInputPoint(@ip)
        view.invalidate
      end
    end

    def onKeyUp(key, repeat, flags, view)
      if key == CONSTRAIN_MODIFIER_KEY
        @switch_edge_side = false
        view.invalidate
      end
    end
    
    def draw(view)
      if(@active_face)
        OffsetCut.preview(view, @active_face.offsetPoints(self.offset))
      end
    end

    def activeFaceFromInputPoint(in_inputPoint)
      #Sketchup::set_status_text "active N="+@N.to_s, SB_VCB_LABEL
      face = nil
      edge_from_input_point = in_inputPoint.edge
      face_from_input_point = in_inputPoint.face
      
      # check edge for non-phlatboyz edge
      if edge_from_input_point and not (edge_from_input_point.phlatedge?)
        faces = edge_from_input_point.faces
        if(faces)
          face = faces[@@N % faces.length] if faces.length != 0
        end
      elsif face_from_input_point
        edges = face_from_input_point.edges
        edges_are_phlatboyz = false
        edges.each do | edge |
          edges_are_phlatboyz = (edges_are_phlatboyz or (edge.phlatedge?))
        end
        if not edges_are_phlatboyz
          face = face_from_input_point
        end
      end
      return face
    end

    def offset
      @switch_edge_side ? -offset_distance : offset_distance
    end

    def statusText
      return "Select face. N to select next face. Shift to invert offset"
    end

    private

    def cut_face(face, dist=0)
      new_edges=[] # holds all the new edges created throughout
      cur_pt = nil
      last_pt = nil
      verts = face.outer_loop.vertices
      0.upto(verts.length) do |i|
        Sketchup.active_model.start_operation "Creating Offset Face", true, true
        a = i % verts.length
        arc_ar = nil
        last_pt = cur_pt if !cur_pt.nil?
        cur_pt = offsetPoint(dist, face, [verts[a-1].position, verts[a].position, verts[a-(verts.length-1)].position])
        next if last_pt.nil? || cur_pt.nil?
        ne = (face.parent.entities.add_edges last_pt, cur_pt)
        next if ne.nil?
        new_edges += ne
        Sketchup.active_model.start_operation "Cutting offset edge", true, true
        phlatcuts = cut_class.cut([ne[0]])
        Sketchup.active_model.commit_operation

        e_last = verts[a].common_edge verts[a-1]
        if ((!e_last.nil?) && (!e_last.curve.nil?) && (e_last.curve.kind_of? Sketchup::ArcCurve))
          c = e_last.curve

          pt1 = c.first_edge.vertices.first.position
          pt2 = c.last_edge.vertices.last.position
          if pt1 == pt2 then #it's a circle
            g3 = false
          else
            pt3 = Geom::Point3d.new [((pt1.x+pt2.x)/2), ((pt1.y+pt2.y)/2), 0]
            g3 = !(Geom.point_in_polygon_2D(pt3, face.outer_loop.vertices, false))
          end
          phlatcuts.each { |phlatcut| phlatcut.define_arc((c.radius + dist), (c.end_angle - c.start_angle), g3) }
        end
        Sketchup.active_model.commit_operation
      end

      ret_f = nil
      new_edges.each { |e| 
        Sketchup.active_model.start_operation "Finding faces", true, true
        e.find_faces
        Sketchup.active_model.commit_operation
      }
      if new_edges.length > 0
        new_edges.first.faces.each { |f|
          ret_f = f if (f.outer_loop.edges.include? new_edges.first)
        }
      end
      if (ret_f)
        Sketchup.active_model.start_operation "Setting face material", true, false
        ret_f.material = "Hole"
        Sketchup.active_model.commit_operation
        return ret_f.outer_loop.edges
      else
        return new_edges
      end
    end

    def offsetPoint(dist, face, points=[])
      pi=Math::PI
      return nil if (not ((dist.class==Fixnum || dist.class==Float || dist.class==Length) && dist!=0))
      vec1=(points[1]-points[2]).normalize!
      vec2=(points[1]-points[0]).normalize!
      vec3=(vec1+vec2).normalize
      return nil if !vec3.valid?
       
      ang=vec1.angle_between(vec2)/2
      ang=pi/2 if vec1.parallel?(vec2)
      vec3.length=dist/Math::sin(ang)
      t=Geom::Transformation.new(vec3)
      pt = points[1].transform(t)
      # the point should be outside the face if dist is greater than 0
      # 1 = point inside 8 = point outside
      pt_class = face.classify_point(pt)
      pt = points[1].transform(Geom::Transformation.new(vec3.reverse)) if (((pt_class == 1) && (dist > 0)) || ((pt_class == 8) && (dist < 0)))
      return pt
    end

  end

  class InsideCutTool < CutTool
    def getContextMenuItems
      return PhlatScript.getString("Inside Edge")
    end

    def offset_distance
      return -@bit_diameter/2
    end

    def cut_class
      return InsideCut
    end
  end

  class OutsideCutTool < CutTool
    def getContextMenuItems
      return PhlatScript.getString("Outside Edge")
    end

    def offset_distance
      return @bit_diameter/2
    end

    def cut_class
      return OutsideCut
    end
  end

end